
/*

 P0   <-----1 PB0      PA0 20-------> BattJP3
 P1   <-----2 PB1   T  PA1 19-------> Power Button / VRef Enable
 P2   <-----3 PB2   i  PA2 18-------> BattJP2
 P3   <-----4 PB3   n  PA3 17-------> VRef
            5 VCC   y AGND 16
            6 GND   8 AVCC 15
 VCC  <-----7 PB4   6  PA4 14-------> GND
 Boost<-----8 PB5   1  PA5 13-------> Battery Sense
 Vfb  <-----9 PB6      PA6 12-------> Boost Enable (Active High)
           10 RESET    PA7 11-------> Data In
                               
fuses: BODLEVEL=111 CKDIV8=1 SUT=10 CKSEL=0010 hfuse=11011111 (0xdf) lfuse=11100010 (0xe2)
*/

#define F_CPU 8000000
#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include <string.h>

unsigned char leds_off;
unsigned short unique_id;
unsigned char unique_id_counter;

/*
  to do:
    * increase # of transmissions
    * fix flickering LEDs when battery meter flashes
    * check range
*/

static unsigned short get_boosted_voltage_reading() {
  union {
    unsigned char bytes[2];
    unsigned short word;
  } reading;

  // select correct analog input
  ADMUX = _BV(REFS0)|_BV(MUX3)|_BV(MUX0);
  // turn on external voltage reference
  DDRA |= _BV(PA1);
  // turn on ADC and start a conversion
  ADCSRA = _BV(ADEN)|_BV(ADPS2)|_BV(ADPS1);
  ADCSRA |= _BV(ADSC);
  // wait for conversion to finish
  while( ADCSRA&_BV(ADSC) )
    ;
  // turn off external voltage reference
  DDRA &= ~_BV(PA1);
  // get ADC reading
  reading.bytes[0] = ADCL;
  reading.bytes[1] = ADCH;
  return reading.word;
}

#define ADC_FULL_READING 18.0
static unsigned char get_battery_limits(unsigned short* min, unsigned short* max, unsigned short* turnoff) {
  unsigned char state;
  PORTA |= _BV(PA0)|_BV(PA2);
  asm volatile("nop");
  asm volatile("nop");
  asm volatile("nop");
  asm volatile("nop");
  asm volatile("nop");
  asm volatile("nop");
  state = PINA;
  PORTA &= ~(_BV(PA0)|_BV(PA2));
  *turnoff = (unsigned short)( 3.0 * 1024 / ADC_FULL_READING);
  if( !(state&_BV(PA2)) ) {
    if( !(state&_BV(PA0)) ) {
      // 4 x AA NiMH
      *min = (unsigned short)( 3.6 * 1024 / ADC_FULL_READING);
      *max = (unsigned short)( 4.8 * 1024 / ADC_FULL_READING);
      *turnoff = (unsigned short)( 3.4 * 1024 / ADC_FULL_READING);
      return 1;
    } else {
      // 9V alkaline
      *min = (unsigned short)( 6.0 * 1024 / ADC_FULL_READING);
      *max = (unsigned short)( 9.0 * 1024 / ADC_FULL_READING);
      return 2;
    }
  } else {
    if( !(state&_BV(PA0)) ) {
      // 4 x AA alkaline
      *min = (unsigned short)( 4.0 * 1024 / ADC_FULL_READING);
      *max = (unsigned short)( 6.0 * 1024 / ADC_FULL_READING);
      return 3;
    } else {
      DDRA |= _BV(PA2);
      PORTA |= _BV(PA0);
      state = PINA;
      DDRA &= ~_BV(PA2);
      PORTA &= ~_BV(PA0);

      if( state&_BV(PA0) ) {
        // 12V lead-acid
        *min = (unsigned short)(11.8 * 1024 / ADC_FULL_READING);
        *max = (unsigned short)(12.8 * 1024 / ADC_FULL_READING);
        *turnoff = (unsigned short)(11.5 * 1024 / ADC_FULL_READING);
        return 4;
      } else {
        // 9V NiMH
        *min = (unsigned short)( 5.4 * 1024 / ADC_FULL_READING);
        *max = (unsigned short)( 7.2 * 1024 / ADC_FULL_READING);
        *turnoff = (unsigned short)(5.1 * 1024 / ADC_FULL_READING);
        return 5;
      }
    }
  }
}

// return battery level as a number 0-15 where 0=flat and 15=full
static unsigned char get_battery_level(unsigned char* shutdown) {
  unsigned short min, max, turnoff;
  union {
    unsigned char bytes[2];
    unsigned short word;
  } reading;

  // select correct analog input
  ADMUX = _BV(REFS0)|_BV(MUX2);
  // turn on external voltage reference
  DDRA |= _BV(PA1);
  // turn on ADC and start a conversion
  ADCSRA = _BV(ADEN)|_BV(ADPS2)|_BV(ADPS1);
  ADCSRA |= _BV(ADSC);
  // work out minimum and maximum expected battery voltages depending upon state of BattJP2 and BattJP3
  get_battery_limits(&min, &max, &turnoff);
  // wait for conversion to finish
  while( ADCSRA&_BV(ADSC) )
    ;
  // turn off external voltage reference
  DDRA &= ~_BV(PA1);
  // get ADC reading
  reading.bytes[0] = ADCL;
  reading.bytes[1] = ADCH;
//  reading.word += 17; // compensate for drop across transistor which switches battery voltage (under typical conditions)
  if( shutdown )
    *shutdown = (reading.word <= turnoff);
  // convert raw reading to 0-15 battery state based on minimum and maximum expected readings
  if( reading.word < min )
    return 0;
  else if( reading.word > max )
    return 15;
  else
    return ((reading.word - min) * 15 + ((max - min)>>1)) / (max - min);
}

static unsigned short get_boosted_voltage() {
  unsigned short sum = 0;
  unsigned char i;
  for( i = 0; i < 4; ++i )
    sum += get_boosted_voltage_reading();
  return sum>>2;
}

unsigned char recv_state;
signed short recv_timing;
unsigned long recv_data, last_recv_data = 0xFFFFFFFF;
volatile unsigned char data_packet, data_packet_valid;

static unsigned short read_TCNT0() {
  union {
    unsigned char bytes[2];
    unsigned short word;
  } data;
  data.bytes[0] = TCNT0L;
  data.bytes[1] = TCNT0H;
  return data.word;
}

static void write_TCNT0(unsigned short value) {
  union {
    unsigned char bytes[2];
    unsigned short word;
  } data;
  data.word = value;
  TCNT0H = data.bytes[1];
  TCNT0L = data.bytes[0];
}

/*
static unsigned short read_OCR0() {
  union {
    unsigned char bytes[2];
    unsigned short word;
  } data;
  data.bytes[0] = OCR0A;
  data.bytes[1] = OCR0B;
  return data.word;
}
*/

static void write_OCR0(unsigned short value) {
  union {
    unsigned char bytes[2];
    unsigned short word;
  } data;
  data.word = value;
  OCR0B = data.bytes[1];
  OCR0A = data.bytes[0];
}

static unsigned char CRC7(unsigned char* data, unsigned char num_bytes) {
  unsigned char crc, datum, i;
  crc = 0; 
  while( num_bytes ) {
    datum = *data;
    for( i=0; i<8; ++i ) {
      crc <<= 1;
      if( (datum&0x80)^(crc&0x80) )
        crc ^= 9;
      datum <<= 1;
    }
    ++data;
    --num_bytes;
  }
  return crc & 0x7f;
}

static void finish_recv() {
  unsigned char temp[3];
  unsigned char unique_id_matches;
  recv_state = 0;
  TIMSK &= ~_BV(OCIE0A);
  temp[0] = recv_data;
  temp[1] = recv_data>>8;
  temp[2] = (recv_data>>16)&0x7f;
  unique_id_matches = (((recv_data>>8)&0x7fff) == unique_id);
  if( recv_data != last_recv_data && CRC7(temp, 3) == (recv_data>>23) && (unique_id_counter < 5 || unique_id_matches) ) {
    if( unique_id_counter < 5 ) {
      if( unique_id_matches ) {
        ++unique_id_counter;
      } else {
        unique_id = ((recv_data>>8)&0x7fff);
        unique_id_counter = 0;
      }
    }
    data_packet = recv_data&0xFF;
    data_packet_valid = 1;
    last_recv_data = recv_data;
  }
}

static unsigned short abs(signed short val) {
  return val < 0 ? -val : val;
}

// each bit is around 400us which is 3200 cycles at 8MHz
#define BIT_LENGTH 3200
#define SLOP 800
#define SLOP2 (SLOP*2)
#define PACKET_BITS 32

/*
  Packet format:

              1111111111222222222233
    01234567890123456789012345678901
    11aaaabbbbcccccccccccccccddddddd
    ....____....____....____....____
    
    a = rudder position (4 bits)
    b = battery level (4 bits)
    c = unique ID (15 bits)
    d = checksum (7 bits)
*/

ISR(TIMER0_COMPA_vect) {
  unsigned char pin_state = (PINA&_BV(PA7)) != 0;
  if( recv_state == PACKET_BITS+2 )
    finish_recv();
  else
    recv_state = pin_state;
#ifdef DEBUG
  PORTB = 0xf;
#endif
  TIMSK &= ~_BV(OCIE0A);
}

ISR(PCINT_vect) {
  unsigned char pin_state = (PINA&_BV(PA7)) != 0;

  switch(recv_state) {
  case 0:
    if( pin_state ) {
      write_TCNT0(0);
      recv_state = 1;
//      PORTB = 8;
    }
    break;
  case 1:
    if( !pin_state ) {
      recv_timing = read_TCNT0();
      write_TCNT0(0);
      if( recv_timing >= BIT_LENGTH-SLOP && recv_timing < BIT_LENGTH+SLOP ) {
        ++recv_state;
      } else {
        recv_state = 0;
//        PORTB = 0xf;
      }
    }
    break;
  case 2:
  case 4:
    if( pin_state ) {
      signed short temp = read_TCNT0();
      write_TCNT0(0);
      if( abs(temp - recv_timing) < SLOP2 ) {
        if( ++recv_state == 5 ) {
          leds_off = 20;
#ifdef DEBUG
          PORTB = 8;
#endif
          recv_data = 0;
          write_OCR0(recv_timing*2+SLOP2*3);
          TIFR |= _BV(OCF0A);
          TIMSK |= _BV(OCIE0A);
        } else {
          recv_timing = (recv_timing + temp)>>1;
        }
      } else {
        recv_state = 1;
      }
    }
    break;
  case 3:
    if( !pin_state ) {
      signed short temp = read_TCNT0();
      write_TCNT0(0);
      if( abs(temp - recv_timing) < SLOP2 ) {
        ++recv_state;
      } else {
        recv_state = 0;
//        PORTB = 0xf;
      }
    }
    break;
  default:
    {
      signed short temp = read_TCNT0();
      if( /*temp > recv_timing-SLOP2 && */temp < recv_timing+SLOP2 ) {
        recv_data |= 0x20000000;
#ifdef DEBUG
        PORTB = 0;//(recv_data&0x20000000) ? 8 : 0xf;
#endif
//        write_TCNT0(recv_timing);
      } else {
        write_TCNT0(0);
#ifdef DEBUG
        PORTB = 8;//(recv_data&0x20000000) ? 8 : 0xf;
#endif
        if( ++recv_state == PACKET_BITS+3 ) {
          finish_recv();
        } else {
          recv_data >>= 1;
        }
      }
    }
    break;
  }
}

static unsigned char battery_reading_to_level(unsigned char reading) {
  if( reading < 7 )
    if( reading < 4 )
      return 0;
    else
      return 1;
  else
    if( reading < 13 )
      if( reading < 10 )
        return 2;
      else
        return 3;
    else
      return 4;
}

int main(void) {
  unsigned char duty = 8, which = 0xf, flash = 0, batt = 0, button = 0, pressed = 1, dimming = 1, dim = 0, skip = 0, skip2 = 0, ignore;
  unsigned char batt_flash = 0, batt_flash_counter = 0;
  const unsigned short target = (unsigned short)(12.0 * 1024 / ADC_FULL_READING + 0.5);
//  unsigned short target;
  const unsigned short min_voltage = (unsigned short)(1.0 * 1024 / ADC_FULL_READING + 0.5);
  unsigned short counter = 0, reading, off_timer = 0;
  unsigned char battery_level, battery_levels[8], cur_battery_level = 0, battery_level_counter = 1, i, shutdown;
  unsigned short off_timeout;
  unsigned char off_timeout_high;
  unsigned char remote_battery, remote_battery_timer = 0;

  DDRA = _BV(PA6);
  DDRB = _BV(PB5)|_BV(PB3)|_BV(PB2)|_BV(PB1)|_BV(PB0);
  PORTB = 0xf;

  // set up ADC: select PB6 (ADC9) as ADC input and PA3 (AREF) as voltage reference
  ADMUX = _BV(REFS0)|_BV(MUX3)|_BV(MUX0);
  // disable digital input for AREF and ADC9
  DIDR0 = _BV(AREFD);
  DIDR1 = _BV(ADC9D);

  if( !(PINA&_BV(PA1)) ) {
    // calibration mode
    ADMUX = _BV(REFS0)|_BV(MUX3)|_BV(MUX0);
    // turn on external voltage reference
    DDRA |= _BV(PA1);
    // turn on ADC and start a conversion
    ADCSRA = _BV(ADEN)|_BV(ADPS2)|_BV(ADPS1);
    ADCSRA |= _BV(ADSC);

    for( ; counter < 30000; ++counter )
      _delay_ms(1.0);

    ADMUX = 0;
    DDRA &= ~_BV(PA1);
    ADCSRA = 0;
    counter = 0;
  }

  GIMSK |= _BV(PCIE1);

  while(1) {
    PCMSK0 = _BV(PCINT1);
    PCMSK1 = 0;
    sei();
    PORTA |= _BV(PA1);
    MCUCR |= _BV(SE)|_BV(SM1);
    asm volatile("sleep");
    if( PINA&_BV(PA1) ) {
      // button was not pressed, go back to sleep
      continue;
    }
    MCUCR = 0;
    PORTA &= ~_BV(PA1);

    // set up timer/counter 1 for PWM via OCD1 (PB5) at 32MHz
    PLLCSR = /*_BV(LSM)|*/_BV(PLLE);
    _delay_ms(0.1);
    while( !(PLLCSR&_BV(PLOCK)) )
      ;
    PLLCSR |= _BV(PCKE);

    TCCR1B = _BV(CS10);
    TCCR1C = _BV(COM1D1)|_BV(PWM1D);
    OCR1D = duty;

    // set up timer and pin change interrupt for 433.9MHz data reception
    TCCR0A = _BV(TCW0);
    TCCR0B = _BV(CS00);
    recv_state = 0;
    PCMSK0 = _BV(PCINT7);

    PORTA |= _BV(PA6);
    ignore = 1;

    for( i = 0; i < 8; ++i ) {
      _delay_ms(1.0);
      battery_levels[i] = get_battery_level(0);
    }

    off_timeout = 0;
    off_timeout_high = 0;
    while(1) {
      if( ++off_timeout == 0 && ++off_timeout_high == 7 ) // about ten minutes after last activity
        break;
      if( --battery_level_counter == 0 ) {
        battery_levels[cur_battery_level] = get_battery_level(&shutdown);
        if( shutdown )
          break;
        cur_battery_level = (cur_battery_level+1)&7;
        if( !remote_battery_timer ) {
          battery_level = 4;
          for( i = 0; i < 8; ++i )
            battery_level += battery_levels[i];
          battery_level >>= 3;
          batt = battery_reading_to_level(battery_level);
          batt_flash = (battery_level == 0);
        } else {
          --remote_battery_timer;
        }
      }

      PORTA |=  _BV(PA1);
      button = (button<<1)|((PINA&_BV(PA1))>>1);
      PORTA &= ~_BV(PA1);
      reading = get_boosted_voltage();
//      target = leds_off || which == 0xf ? (unsigned short)(7.5 * 1024 / ADC_FULL_READING + 0.5) : (unsigned short)(12.0 * 1024 / ADC_FULL_READING + 0.5);
      if( reading < target-(leds_off ? 10 : 4) && duty < 160/*172*/ ) {
        ++duty;
        OCR1D = duty;
      } else if( reading > target+(leds_off ? 10 : 4) && duty > 0 ) {
        --duty;
        OCR1D = duty;
      }
      if( data_packet_valid ) {
        data_packet_valid = 0;

        if( (data_packet&7) == 7 ) {
          if( which == 3 )
//          which = 3;
            flash = 0;
        } else {
          which = (data_packet&7);
          flash = which == 3;
        }

        remote_battery = (data_packet>>4)&15;
        remote_battery_timer = 10;
        batt = battery_reading_to_level(remote_battery);
        batt_flash = 2;

        counter = 0;
        off_timeout = 0;
        off_timeout_high = 0;
//        if( ++which >= 7 )
//          which = 0;
      }
      if( dim++ < dimming || leds_off ) {
#ifndef DEBUG
        PORTB = 0xf;
#endif
        _delay_ms(1);
        if( leds_off )
          --leds_off;
        if( batt_flash )
          ++batt_flash_counter;
        if( remote_battery_timer )
          ++batt_flash_counter;
        if( flash ) {
          if( ++counter == 300 )
            counter = 0;
        }
      } else {
        unsigned char temp = ((batt+skip2)>>1);
#ifndef DEBUG
        if( batt_flash && ++batt_flash_counter >= 150 )
          PORTB = 0xf;
        else
          PORTB = temp+7;
#endif
        if( remote_battery_timer )
          ++batt_flash_counter;
        skip2 ^= 1;
        if( temp == 0 )
          _delay_ms(0.2);
        else if( temp == 1 )
          _delay_ms(0.15);
        else
          _delay_ms(0.25);

        if( flash && ++counter >= 150 ) {
#ifndef DEBUG
          PORTB = 0xf;
#endif
        } else {
          if( which < 4 ) {
            if( ++skip&1 ) {
#ifndef DEBUG
              PORTB = 0xf;
#endif
            } else {
#ifndef DEBUG
              PORTB = which;
#endif
            }
          } else {
#ifndef DEBUG
            PORTB = which;
#endif
          }
        }
        _delay_ms(0.5);
        dim = 0;
        if( counter == 300 )
          counter = 0;
      }

      if( button == 0 && !pressed ) {
        off_timer = 1000;
        pressed = 1;
        off_timeout = 0;
        off_timeout_high = 0;
      } else if( button == 0xff && pressed ) {
        pressed = 0;
        if( !ignore ) {
          if( ++dimming == 2 )
            ++dimming;
          else if( dimming == 4 )
            dimming = 0;
        }
        ignore = 0;
      } else if( pressed && off_timer && --off_timer == 0 ) {
        button = 0;
        break;
      }
    }

    // turn off PWM & PLL
    TCCR1B = 0;
    TCCR1C = 0;
    PLLCSR = 0;
    OCR1D = 0;
 
    // turn off boost circuit
    PORTA &= ~_BV(PA6);

    // turn off LEDs and forget state
    which = 0xf;
    PORTB = 0xf;

    // wait for boosted voltage to decay
    while( get_boosted_voltage() > min_voltage )
      ;

    // bring LED control lines low
    PORTB = 0x0;

    // turn off ADC
    ADCSRA = 0;

    // wait for button to be released
    PORTA |= _BV(PA1);
    do {
      _delay_ms(1.0);
      button = (button<<1)|((PINA&_BV(PA1))>>1);
    } while( button != 0xff );
  }
  return 0;
}
